// filters.ino : filtering primitives used by the ClassifierDemo sketch.
// No copyright, 2020, Garth Zeglin.  This file is explicitly placed in the public domain.

//================================================================
// Suppress a specific value in an input stream.  One integer of state is required.
int suppress_value(int input, int value)
{
  static int previous = 0;
  if (input != value) previous = input;
  return previous;
}

//================================================================
// Debounce an integer stream by suppressing changes from the previous value
// until a specific new value has been observed a minimum number of times. Three
// integers of state are required.

int debounce(int input, int samples)
{
  static int current_value = 0;
  static int new_value = 0;
  static int count = 0;

  if (input == current_value) {
    count = 0;
  } else {
    if (count == 0) {
      new_value = input;
      count = 1;
    } else {
      if (input == new_value) {
	count += 1;
	if (count >= samples) {
	  current_value = new_value;
	  count = 0;
	}
      } else {
	new_value = input;
	count = 1;
      }
    }
  }
  return current_value;
}

//================================================================
// Floating-point version of map().  The standard Arduino map() function only
// operates using integers; this extends the idea to floating point.  The
// Arduino function can be found in the WMath.cpp file within the Arduino IDE
// distribution.  Note that constrain() is defined as a preprocessor macro and
// so doesn't have data type limitations.

float fmap(float x, float in_min, float in_max, float out_min, float out_max) {
  float divisor = in_max - in_min;
  if (divisor == 0.0) {
    return out_min;
  } else {
    return (x - in_min) * (out_max - out_min) / divisor + out_min;
  }
}
//================================================================
// Low-Pass Butterworth IIR digital filter, generated using filter_gen.py.
// Sampling rate: 10 Hz, frequency: 1.0 Hz.
// Filter is order 4, implemented as second-order sections (biquads).
// Reference: https://docs.scipy.org/doc/scipy/reference/generated/scipy.signal.butter.html
float lowpass(float input)
{
  float output = input;
  {
    static float z1, z2; // filter section state
    float x = output - -1.04859958*z1 - 0.29614036*z2;
    output = 0.00482434*x + 0.00964869*z1 + 0.00482434*z2;
    z2 = z1;
    z1 = x;
  }
  {
    static float z1, z2; // filter section state
    float x = output - -1.32091343*z1 - 0.63273879*z2;
    output = 1.00000000*x + 2.00000000*z1 + 1.00000000*z2;
    z2 = z1;
    z1 = x;
  }
  return output;
}

//================================================================
// Trajectory estimation filter generated using trajfit_gen.py.
// Based on Savitzky-Golay polynomial fitting filters.
// Sampling rate: 10 Hz.
// The output array will contain the trajectory parameters representing the signal
// at the current time: [position, velocity, acceleration], with units of [1, 1/sec, 1/sec/sec].
// Reference: https://docs.scipy.org/doc/scipy/reference/generated/scipy.signal.savgol_coeffs.html
void trajfit(float input, float output[3])
{
  const float coeff[3][5] = 
    {{  0.085714,  -0.142857,  -0.085714,   0.257143,
        0.885714},
     {  3.714286,  -3.857143,  -5.714286,  -1.857143,
        7.714286},
     { 28.571429, -14.285714, -28.571429, -14.285714,
       28.571429}};
  static float ring[5]; // buffer for recent time history
  static unsigned oldest = 0; // index of oldest sample

  // save the new sample by overwriting the oldest sample
  ring[oldest] = input;
  if (++oldest >= 5) oldest = 0;

  // iterate over the coefficient rows
  unsigned index = oldest;
  for (int i = 0; i < 3; i++) {
    output[i] = 0.0; // clear accumulator

    // Iterate over the samples and the coefficient rows.  The index cycles
    // around the circular buffer once per row.
    for (int j = 0; j < 5; j++) {
      output[i] += coeff[i][j] * ring[index];
      if (++index >= 5) index = 0;
    }
  }
}
//================================================================
