#!/usr/bin/env python3
"""motion_detector.py : self-contained camera video processing example
"""
import argparse, time
import numpy as np

# https://docs.opencv.org/modules/highgui/doc/highgui.html
# https://docs.opencv.org/modules/highgui/doc/reading_and_writing_images_and_video.html
# https://docs.opencv.org/modules/video/doc/motion_analysis_and_object_tracking.html
import cv2 as cv

from pythonosc import osc_message_builder
from pythonosc import udp_client

#================================================================
if __name__ == "__main__":
    parser = argparse.ArgumentParser( description = "Process a video stream from the default camera.")
    parser.add_argument( '--camera', default=0, type=int, help="Select camera by number (default: %(default)s).")
    parser.add_argument( '-s','--saturation', action='store_true', help='Use only saturation channel.')
    parser.add_argument( '-b','--background', action='store_false', help='Disable background subtraction.')
    parser.add_argument( '-q','--quiet', action='store_true', help="Run without viewer window or console output.")
    parser.add_argument( '--debug', action='store_true', help='Enable debugging output to log file.' )
    parser.add_argument( '--verbose', action='store_true', help='Enable even more detailed logging output.' )
    parser.add_argument("--ip", default="127.0.0.1",  help="IP address of the OSC receiver (default: %(default)s).")
    parser.add_argument("--port", type=int, default=22000,  help="OSC receiver UDP port (default: %(default)s).")
    args = parser.parse_args()

    capture = cv.VideoCapture(args.camera)
    if not capture.isOpened():
        print("Unable to open camera, exiting.")
        exit(1)

    if not args.quiet:
        cv.namedWindow("motion detector")

    if args.background:
        bgs = cv.createBackgroundSubtractorMOG2()

    client = udp_client.SimpleUDPClient(args.ip, args.port)

    def repeat():
        success, frame = capture.read()
        if success:
            if args.saturation:
                # transform the image
                frame = cv.cvtColor(frame, cv.COLOR_BGR2HSV)
                # pick out saturation
                frame = frame[:,:,1]

            if args.background:
                foreground_mask = bgs.apply(frame)
                frame = foreground_mask

            # reduce the lower half of the frame to a tiny 1D image
            rows, cols = frame.shape[0:2]
            motion = cv.resize(frame[rows//2:][:], (16, 1))

            # replace the original lower half with a blocky 1D image
            frame[rows//2:][:] = cv.resize(motion, (cols, rows//2), interpolation=cv.INTER_NEAREST)

            # print ("Showing image...")
            if not args.quiet:
                cv.imshow("viewer", frame)

            # send black and white values as an OSC message
            if args.background:
                msg = [int(i) for i in motion[:][0]]
                # print("Sending", msg, type(msg[0]))
                client.send_message("/motion", msg)

        else:
            if args.verbose:
                print ("no image")

    try:
        while True:
            repeat()

            # Calling waitKey allows the window system event loop to run and update
            # the display.  The waitKey argument is in milliseconds.
            if cv.waitKey(33) & 0xFF == ord('q'):
                break

    except KeyboardInterrupt:
        print ("User quit.")

    capture.release()
