#!/usr/bin/env python3

# main.py : top-level script to run a suitcase testbed show
# No copyright, 2021, Garth Zeglin.  This file is explicitly placed in the public domain.

# This provides a skeleton Webots API to connect a Webots controller object to 
# physical hardware.  Unlike Webots, it uses threads to run the controller as
# a normal Python object within a single process.

# Standard Python modules.
import os, sys, time, threading, argparse

# Import pySerial module for communicating with an Arduino.  https://pythonhosted.org/pyserial/
import serial

# Import the skeleton Webots API from the current folder.
import controller

# Import the controller. The suitcase.py file was copied from a Webots project tree.
import suitcase

################################################################
# Define wrapper classes which inherit from both a Webots robot controller and a
# specialized physical interface class.  The controller object inherits
# controller.Robot with common API code, but some methods are implemented
# instead by the SuitcaseRobot interface class.

class RealSuitcase(suitcase.Suitcase, controller.SuitcaseRobot, threading.Thread):
    def __init__(self):
        print("Entering RealSuitcase.__init__ and initializing superclasses")
        super(RealSuitcase, self).__init__()
        self.daemon = True
        
    def run(self):
        print("Starting suitcase.")
        self._lock.acquire()
        super(RealSuitcase, self).run()
        
################################################################
class Arduino(threading.Thread):
    """Connect an Arduino serial port to a controller object.  The input and output
    is assumed to be a text line of space-delimited integers. The specific data
    format of both input is determined by the controller class and the Arduino
    sketch.  This process runs as a thread so it can easily block on the input.
    """
    def __init__(self, portname, controller):
        print("Entering Arduino.__init__ and initializing superclasses")
        super(Arduino, self).__init__()

        self.controller = controller
        self.portname = portname
        self.port = serial.Serial(portname, baudrate=115200, timeout=1.0)
        self.running = True
        
    def run(self):
        # wait briefly for the Arduino to complete waking up
        time.sleep(0.2)
        
        print(f"Entering Arduino event loop for {self.portname}.")
        while(self.running):
            line = self.controller.get_output_string()
            self.port.write(line.encode())
            print("Sending to Arduino:    " + line.strip())

            input = self.port.readline().decode('ascii')
            print("Received from Arduino: " + input.strip())
            self.controller.process_sensor_string(input)

    def safe_exit(self):
        self.running = False
        self.port.write(b"enable 0\n")        
        
################################################################        
# The following section is run when this is loaded as a script.
if __name__ == "__main__":

    # Initialize the command parser.
    parser = argparse.ArgumentParser( description = """Control script to connect Webots controller to StepperWinch firmware on an Arduino.""")
    parser.add_argument( '-p', '--port', default='/dev/cu.usbmodem1444301',
                         help='Specify the name of the Arduino serial port device (default is %(default)s.')

    # Parse the command line, returning a Namespace.
    args = parser.parse_args()
    
    # Create the suitcase control object.
    suitcase = RealSuitcase()

    # Create the Arduino interface which communicates with physical hardware.
    # The serial device address will need to be modified for your specific computer.
    suitcase_hw = Arduino(args.port, suitcase)

    # Start the control thread.
    suitcase.start()

    # Start the hardware interface thread.
    suitcase_hw.start()

    # Wait for the hardware thread to finish.
    try:
        suitcase_hw.join()

    except KeyboardInterrupt:
        suitcase_hw.safe_exit()
        print("Main thread received user interrupt, exiting.")
    

