#!/usr/bin/env python3
"""light-fan-ensemble-demo.py : demonstration of scripting action across all theater lighting units
"""
import argparse, time, logging
import numpy as np

from pythonosc import osc_message_builder
from pythonosc import udp_client

import stage.config
import stage.network

# import common logging functions
import stage.logconfig

# initialize logging for this module
log = logging.getLogger('demo')

#================================================================
if __name__ == "__main__":

    # set up logging
    stage.logconfig.open_log_file('logs/light-fan-ensemble-demo.log')
    log.info("Starting demo show.")

    parser = argparse.ArgumentParser( description = "Simple scripted show.")
    parser.add_argument("--ip", default=stage.config.theater_IP,
                        help="IP address of the OSC receiver (default: %(default)s).")
    stage.logconfig.add_logging_args(parser)
    args = parser.parse_args()

    # Modify logging settings as per common arguments.
    stage.logconfig.configure_logging(args)

    # create OSC clients to send messages to the theater system
    network = stage.network.TheaterNetwork(args)

    # pulse each RGBA fixture with a specific color
    color_table = {'rgba1' : [0, 0, 255, 0],
                   'rgba2' : [255, 0, 0, 0],
                   'rgba3' : [0, 0, 0, 255],
                   'rgba4' : [0, 255, 0, 255],}

    for fixture in ['rgba1', 'rgba2', 'rgba3', 'rgba4']:
        network.lights.send_message("/fixture", [fixture] + color_table[fixture])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)
        network.lights.send_message("/fixture", [fixture, 0, 0, 0, 0])

    # activate each mono light
    for fixture in ['lamp1', 'lamp2', 'lamp3', 'lamp4']:
        network.lights.send_message("/fixture", [fixture, 255])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(1)

    # activate each fan one at a time
    for fixture in ['fan1', 'fan2', 'fan3', 'fan4']:
        network.lights.send_message("/fixture", [fixture, 255])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)

    # deactivate each fan one at a time
    for fixture in ['fan1', 'fan2', 'fan3', 'fan4']:
        network.lights.send_message("/fixture", [fixture, 0])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)

    # deactivate mono lights
    for fixture in ['lamp1', 'lamp2', 'lamp3', 'lamp4']:
        network.lights.send_message("/fixture", [fixture, 0])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(1)

    # pulse each RGBA fixture full on
    for fixture in ['rgba1', 'rgba2', 'rgba3', 'rgba4']:
        network.lights.send_message("/fixture", [fixture, 255, 255, 255, 255])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)
        network.lights.send_message("/fixture", [fixture, 0, 0, 0, 0])

    log.info("Exiting demo show.")
