#!/usr/bin/env python3
"""demo-lighting.py : demonstration of scripting lighting changes
"""
import argparse, time, logging
import numpy as np

from pythonosc import osc_message_builder
from pythonosc import udp_client

import stage.config
import stage.network

# import common logging functions
import stage.logconfig

# initialize logging for this module
log = logging.getLogger('demo')

#================================================================
if __name__ == "__main__":

    # set up logging
    stage.logconfig.open_log_file('logs/demo-lighting.log')
    log.info("Starting demo-lighting.py")

    parser = argparse.ArgumentParser( description = "Simple scripted show.")
    parser.add_argument("--ip", default=stage.config.theater_IP,
                        help="IP address of the OSC receiver (default: %(default)s).")
    stage.logconfig.add_logging_args(parser)
    args = parser.parse_args()

    # Modify logging settings as per common arguments.
    stage.logconfig.configure_logging(args)

    # create OSC clients to send messages to the theater system
    network = stage.network.TheaterNetwork(args)

    # pulse each RGBA fixture with a specific color
    color_table = {'rgba1' : [0, 0, 255, 0],
                   'rgba2' : [255, 0, 0, 0],
                   'rgba3' : [0, 255, 0, 0],
                   'rgba4' : [0, 0, 0, 255],
                   }

    for fixture in ['rgba1', 'rgba2', 'rgba3', 'rgba4']:
        network.lights.send_message("/fixture", [fixture] + color_table[fixture])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)

    # activate each mono light
    for fixture in ['lamp1', 'lamp2',]:
        network.lights.send_message("/fixture", [fixture, 255])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(1)

    # deactivate rgba lights
    for fixture in ['rgba1', 'rgba2', 'rgba3', 'rgba4']:
        network.lights.send_message("/fixture", [fixture, 0, 0, 0, 0])
        time.sleep(1)

    # deactivate mono lights
    for fixture in ['lamp1', 'lamp2']:
        network.lights.send_message("/fixture", [fixture, 0])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(1)

    # pulse each RGBA fixture full on
    for fixture in ['rgba1', 'rgba2', 'rgba3', 'rgba4']:
        network.lights.send_message("/fixture", [fixture, 255, 255, 255, 255])
        log.debug("Lighting fixture %s", fixture)
        time.sleep(2)
        network.lights.send_message("/fixture", [fixture, 0, 0, 0, 0])

    log.info("Exiting demo-lighting.py")
