#!/usr/bin/env python3
"""demonstrate vision system interaction
"""
import argparse, time, logging
import numpy as np

from pythonosc import osc_message_builder
from pythonosc import udp_client

import stage.config
import stage.network

# import common logging functions
import stage.logconfig

# initialize logging for this module
log = logging.getLogger('demo')

#================================================================
def idle_show_sequence(network):
    network.lights.send_message("/fixture", ['rgba1', 0, 0, 128, 0])
    network.lights.send_message("/fixture", ['rgba3', 0, 0, 128, 0])
    time.sleep(2)
    network.lights.send_message("/fixture", ['rgba1', 0, 0, 0, 0])
    network.lights.send_message("/fixture", ['rgba3', 0, 0, 0, 0])
    time.sleep(1)

#================================================================
def right_show_sequence(network):
    network.lights.send_message("/fixture", ['rgba1', 0, 255, 0, 0])
    time.sleep(2)
    network.lights.send_message("/fixture", ['rgba1', 0, 0, 0, 0])
    time.sleep(1)

#================================================================
def left_show_sequence(network):
    network.lights.send_message("/fixture", ['rgba3', 0, 255, 0, 0])
    time.sleep(2)
    network.lights.send_message("/fixture", ['rgba3', 0, 0, 0, 0])
    time.sleep(1)

#================================================================
if __name__ == "__main__":

    # set up logging
    stage.logconfig.open_log_file('logs/vision-protocol-test.log')
    log.info("Starting vision-protocol-test.py")

    parser = argparse.ArgumentParser( description = "Demonstrate the vision system protocol.")

    parser.add_argument("--ip", default=stage.config.theater_IP,
                        help="IP address of the remote OSC receiver (default: %(default)s).")

    parser.add_argument("--recv", default='0.0.0.0',
                        help="IP address of the local OSC receiver (default: %(default)s).")

    stage.logconfig.add_logging_args(parser)
    args = parser.parse_args()

    # Modify logging settings as per common arguments.
    stage.logconfig.configure_logging(args)

    # create OSC clients to send messages to the theater system
    network = stage.network.TheaterNetwork(args)

    # start a local server to receive messages from the theater vision system
    network.start_vision_client()

    # start event loop
    try:
        while True:
            if network.motion_detected[0]:
                log.info("seeing motion at arm window, modifying performance...")
                right_show_sequence(network)

            elif network.motion_detected[1]:
                log.info("seeing motion at wheel window, modifying performance...")
                left_show_sequence(network)
            else:
                log.info("running idle performance...")
                idle_show_sequence(network)

            time.sleep(1)

    except KeyboardInterrupt:
        log.info("user interrupt, shutting down.")

    log.info("Exiting demo-vision.py")
