#!/usr/bin/env python3
"""winch-demo.py : demonstration of scripting theater actions
"""
import argparse, time, logging
import numpy as np

from pythonosc import osc_message_builder
from pythonosc import udp_client

import stage.config
import stage.network

# import common logging functions
import stage.logconfig

# initialize logging for this module
show_name = 'winch-demo'
log = logging.getLogger(show_name)

#================================================================
if __name__ == "__main__":

    # set up logging
    stage.logconfig.open_log_file('logs/' + show_name + '.log')
    log.info("Starting %s", show_name)

    parser = argparse.ArgumentParser( description = "Simple scripted show.")
    parser.add_argument("--ip", default=stage.config.theater_IP,
                        help="IP address of the OSC receiver (default: %(default)s).")
    stage.logconfig.add_logging_args(parser)
    args = parser.parse_args()

    # Modify logging settings as per common arguments.
    stage.logconfig.configure_logging(args)

    # create OSC clients to send messages to the theater system
    network = stage.network.TheaterNetwork(args)

    # select a single motion unit; each winch set has a separate server
    winch1 = network.motion_server_by_name('winch1')
    winch2 = network.motion_server_by_name('winch2')

    # move each winch axis in turn
    for axis in ['x', 'y', 'z', 'a']:
        winch1.send_message("/spline", [axis, 0.0, 180.0, 180.0])
        time.sleep(0.5)

    # reverse back to start; each spline is relative to current position
    for axis in ['x', 'y', 'z', 'a']:
        winch1.send_message("/spline", [axis, 0.0, -180.0, -180.0])
        time.sleep(0.5)

    # Similar, but with a multi-segment cubic Beizer spline.  Each segment
    # requires three knot values.
    for axis in ['x', 'y', 'z', 'a']:
        winch1.send_message("/spline", [axis, 0.0, 180.0, 180.0,  90.0, -45.0, 0.0])
        time.sleep(0.5)

    # coordinated movement across different winch sets
    for axis in ['x', 'y', 'z', 'a']:
        winch1.send_message("/spline", [axis, 0.0, -180.0, -180.0])
        winch2.send_message("/spline", [axis, 0.0, -180.0, -180.0])
        time.sleep(0.5)

    log.info("Exiting %s", show_name)
