#!/usr/bin/env python3

# merge-MIDI-files.py

# Sample script to merge a set of MIDI files.  This is intended as a script
# template to be customized per application.

# Typical use: merge a set of single-track type 0 MIDI files exported from
# Ableton Live into a multi-track type 1 file.

import sys

# https://mido.readthedocs.io/en/latest/index.html
import mido

#================================================================
# Load an input file.
def load_midi_file(path):
    mid = mido.MidiFile(path)
    print(f"Opened MIDI file {path}: type {mid.type}")
    if mid.type != 0:
        print("Error: this script only supports type 0 (single-track) files.")
        sys.exit(1)
    print("  Track name: %s" % (mid.tracks[0].name))
    return mid

#================================================================
def extract_metadata_track(track):
    """Return a new track containing the metadata common to all tracks."""
    track0 = mido.MidiTrack()
    track0.name = 'metadata'
    for event in track:
        if event.is_meta:
            if event.type in ['time_signature', 'key_signature', 'smpte_offset', 'set_tempo']:
                track0.append(event)

    return track0

#================================================================
def filter_track(track, channel):
    """Process a track for merging, returning a new track.  Filters out
    particular metadata events.  Rewrites channel numbers for message events in
    the track."""

    result = mido.MidiTrack()
    for event in track:
        if event.is_meta:
            if event.type not in ['time_signature', 'key_signature', 'smpte_offset', 'set_tempo']:
                result.append(event)

        else:
            result.append(event.copy(channel=channel))
    return result

#================================================================
#================================================================
# Script execution begins here.

# Load the source files in channel order.
paths = ["channel1.mid", "channel2.mid", "channel3.mid", "channel4.mid", "channel5.mid"]
sources = [load_midi_file(path) for path in paths]

#----------------------------------------------------------------
# Use the first file to generate a metadata track.  Note: no check
# is made to ensure that the subsequent files match.
track0 = extract_metadata_track(sources[0].tracks[0])

# Note: Ableton Live does not include tempo information when exporting clips as
# type 0 MIDI files, it always assumes 120 BPM when generating timing tick
# values.  The following line can be modified to insert your preferred tempo.
file_tempo_in_BPM = 120
track0.append(mido.MetaMessage(type='set_tempo', tempo=mido.bpm2tempo(file_tempo_in_BPM), time=0))

#----------------------------------------------------------------
# Create a new output file to contain all the tracks as individual tracks.
output = mido.MidiFile()
output.ticks_per_beat = sources[0].ticks_per_beat

# Add the metadata track.
output.tracks.append(track0)

#----------------------------------------------------------------
# Process each track and append to the output file.
# Note that channel numbers for each track event are rewritten starting with
# MIDI channel 1 (denoted with zero).

for channel, src in enumerate(sources):
    track = src.tracks[0]
    processed = filter_track(track, channel)
    output.tracks.append(processed)

#----------------------------------------------------------------
# Save the result.
output_path = 'performance.mid'
print(f"Writing output to {output_path}.")
output.save(output_path)

# All done.
